/* $Id: selftest.c,v 1.82 1999/02/24 21:00:18 dhiller Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* User routine for doing self test of E1432 module */

#include "sema.h"

#ifdef HPVXI_DOWNLOAD
#include "math.h"	/* for sqrt() */
#include "diag_aux.h"	/* diag support functions */
#endif

/* #define DEBUG_MODE */	/* uncomment for tighter tolerenaces (debug) */

#define MAX_MODULES_TESTED	15
#define TEST_BLOCKSIZE		256

#define E1432_CLARI_SSHUT		0xfff3
#define E1432_CLARI_FSHUT		0xfff4
#define E1432_CLARI_CLRSHUT		0xfff5
#define E1432_CLARI_1_AZDAC_PRE		0xfff9
#define E1432_CLARI_1_AZDAC_POST	0xfffc
#define E1432_CLARI_2_AZDAC_PRE		0xffe9
#define E1432_CLARI_2_AZDAC_POST	0xffec

#define E1432_SCA_CLARION_1		0	/* sca number of Clarion 1 */
#define E1432_SCA_CLARION_2		2	/* sca number of Clarion 2 */

#define BOTTOM_1M_WIN		0x0003ffff	/* mask off upper 1 M window */

#define E1432_SELFTEST_SOURCE \
  (E1432_SELFTEST_CLARINET | E1432_SELFTEST_CLARINET)


#ifdef DEBUG_MODE
#define OFFSET_LIMIT			.1f
#define INBAND_AMP_LIMIT		.1f
#define TRANSBAND_AMP_LIMIT_POS		.5f
#define TRANSBAND_AMP_LIMIT_NEG		-.2f
#define OFF_LIMIT			.02f
#else
#define OFFSET_LIMIT			.2f
#define INBAND_AMP_LIMIT		.2f
#define TRANSBAND_AMP_LIMIT_POS		1.0f
#define TRANSBAND_AMP_LIMIT_NEG		-.5f
#define OFF_LIMIT			.05f
#endif

/* wait to final ADC read MONEXTMSG in case previous activity dropped it out
   of real time - allows the AAF filter to settle again */
#define E1432_CLARI_ADC_SETTLE	.002f

#define CHK(func) { err = (func); if (err) goto cleanup; }

static const LONGSIZ32 pre_dac[] =
  {E1432_CLARI_1_AZDAC_PRE, E1432_CLARI_2_AZDAC_PRE};
static const LONGSIZ32 post_dac[] =
  {E1432_CLARI_1_AZDAC_POST, E1432_CLARI_2_AZDAC_POST};

static const char gain_str[] = "gain out of range";
static const char offset_str[] = "offset out of range";
static const char oorl_str[] = "measurement too high";
static const char oorh_str[] = "measurement too low";
static const char const_ne_str[] = "constant data near expected value";
static const char const_p_str[] = "data railed positive";
static const char const_n_str[] = "data railed negaitive";
static const char const_z_str[] = "all zero's data";
static const char const_o_str[] = "constant data";
static const char unstable_str[] = "noisy/unstable data";
static const char nds_str[] = "negative squared data";
static const char sso_str[] = "squared sum overflow";
static const char c_sep[] = ", ";
/*static const char s_sep[] = " ";*/
/*static const char all_cal_pass[] = "all";*/
#define VIB_CAL_PASSES 4
static const char cal_pass_lp[] = "low gain & positive cal input";
static const char cal_pass_ln[] = "low gain & negative cal input";
static const char cal_pass_lg[] = "low gain & ground input";
static const char cal_pass_hp[] = "high gain & positive cal input";
static const char cal_pass_hn[] = "high gain & negative cal input";
static const char cal_pass_hg[] = "high gain & ground input";
static const char *const vib_cal_pass[] = { cal_pass_lp, cal_pass_ln,
					    cal_pass_lg, cal_pass_hg };
#define SON_CAL_PASSES 6
static const char *const son_cal_pass[] = { cal_pass_hp, cal_pass_hn,
					    cal_pass_hg, cal_pass_lp,
					    cal_pass_ln, cal_pass_lg };
/*static const char cal_pass_low[] = "low gain compute";*/
/*static const char cal_pass_high[] = "high gain compute";*/

#define ERR_STRLEN 256

#define FW_FAIL ( E1432_FLAG_CAL_FAILURE_FW | E1432_FLAG_CAL_FAILURE_SUB_FW )
#define CAL_FAIL ( E1432_FLAG_CAL_FAILURE_GAIN | E1432_FLAG_CAL_FAILURE_OFFSET \
  | E1432_FLAG_CAL_FAILURE_OORL | E1432_FLAG_CAL_FAILURE_OORH \
  | E1432_FLAG_CAL_FAILURE_CONSTANT | E1432_FLAG_CAL_FAILURE_UNSTABLE )
#define NUM_FAIL ( E1432_FLAG_CAL_FAILURE_NDS | E1432_FLAG_CAL_FAILURE_SSO )
#define CAL_NUM_FAIL ( CAL_FAIL | NUM_FAIL )
#define MEAS_PASS_INFO ( E1432_CAL_FAIL_LOW_GAIN | E1432_CAL_FAIL_HIGH_GAIN )
#define PASS_INFO ( MEAS_PASS_INFO | E1432_CAL_FAIL_PASS_MASK )

#define OVERREADTEST_SRCREV	0x3031UL

static int
chans_in_group(E1432ID hw, SHORTSIZ16 ID)
{
    int chans = 0;
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;

    if ( ID > 0 ) return 1;
    if ( ID == 0 ) return 0;

    /* iterate thru group */
    gn = i1432_get_group_node(hw, ID);
    cn = gn->chanlist;
    while (cn)
    {
	chans++;
        cn = cn->next;
    }
    return chans;
}


SHORTSIZ16 EXPORT
e1432_test_lbus(E1432ID hw, SHORTSIZ16 chan, SHORTSIZ16 mode)
{
    E1432_MODULE_LIST_NODE *mn;
    SHORTSIZ16 error;
    int save_d32;

    if (chan < 0)
	return i1432_print_error(ERR1432_EXPECT_CHANNEL);
    error = i1432_get_module_from_chan(hw, chan, &mn);
    if (error)
	return error;
    save_d32 = mn->d32;

    /* Disable D32 access while running the local bus test.  Things
       mostly work even if we don't do this, but there's no sense in
       taking chances. */
    mn->d32 = 0;
    error = i1432_write_cmd1(hw, chan, E1432_CMD_TEST_LBUS,
			     (LONGSIZ32)mode);

    mn->d32 = save_d32;
    return error;
}


/***************************************************************************
 Installs substrate firmware, unless it has already been successfully
 installed 
 ***************************************************************************/
static SHORTSIZ16
i1432_preinstall(SHORTSIZ16 la)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 install_flags = 0;

    if ( i1432_install_from_mem )
    {
        install_flags = E1432_INSTALL_FROM_MEM | E1432_INSTALL_DECOMPRESS;
    }

    /* need to do this better
    if ( ! (i1432_dnld_fw_state & E1432_DNLD_FW_SUCCESSFUL ) )
    */
    {
        DIAG_PRINTF(2, ("  Downloading %s to LA %d\n", i1432_install_file, la))
        err = e1432_install(1, &la, install_flags, NULL);
    }

    return err;
}


static void
append_err_str(char *str, char *append_str, const char *sep)
{
    int in_len = strlen(str);
    int sep_len = strlen(sep);

    /* allow 2 extra for ", " */
    if ( in_len + sep_len + strlen(append_str) + 2 > ERR_STRLEN ) return;
    if ( in_len > 0 ) (void) strcat(str, sep);
    (void) strcat(str, append_str);
}


static char *
cal_err_str(char *str, LONGSIZ32 failure)
{
    str[0] = '\0';  /* initialize string to "" */

    /* drop in the failures */
    if ( failure & E1432_FLAG_CAL_FAILURE_GAIN )
    {
	append_err_str(str, (char *)gain_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_OFFSET )
    {
	append_err_str(str, (char *)offset_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_OORL )
    {
	append_err_str(str, (char *)oorl_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_OORH )
    {
	append_err_str(str, (char *)oorh_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_CONSTANT )
    {
	/* data available in upper bits, interpret it */
        if ( failure & E1432_CAL_FAIL_CONST_NEAR )
        {
	    append_err_str(str, (char *)const_ne_str, c_sep);
        }
        if ( failure & E1432_CAL_FAIL_CONST_FS_P )
        {
	    append_err_str(str, (char *)const_p_str, c_sep);
        }
        if ( failure & E1432_CAL_FAIL_CONST_FS_N )
        {
	    append_err_str(str, (char *)const_n_str, c_sep);
        }
        if ( failure & E1432_CAL_FAIL_CONST_ZERO )
        {
	    append_err_str(str, (char *)const_z_str, c_sep);
        }
        if ( failure & E1432_CAL_FAIL_CONST_OTHER )
        {
	    append_err_str(str, (char *)const_o_str, c_sep);
        }
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_UNSTABLE )
    {
	append_err_str(str, (char *)unstable_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_NDS )
    {
	append_err_str(str, (char *)nds_str, c_sep);
    }
    if ( failure & E1432_FLAG_CAL_FAILURE_SSO )
    {
	append_err_str(str, (char *)sso_str, c_sep);
    }

    str[0] -= 0x20; /* change first character to upper case */

    return str;
}


static void
cal_print_passes(LONGSIZ32 failure, int sca_type)
{
    const char *const *pass_str;
    int pass, passes;
    int all_passes = 1;

    if ( (failure & PASS_INFO) == 0 ) return;

    switch( sca_type )
    {
    case E1432_SCA_ID_VIBRATO:
	pass_str = vib_cal_pass;
	passes = VIB_CAL_PASSES;
        break;
    case E1432_SCA_ID_SONATA_A:
    case E1432_SCA_ID_SONATA_B:
	pass_str = son_cal_pass;
	passes = SON_CAL_PASSES;
        break;
    default:
        DIAG_PRINTF(2, ("  Unknown sca in cal_print_passes().\n"))
	return;
    }
    for ( pass = 0; pass < passes; pass++ )
    {
	if ( (failure & (E1432_CAL_FAIL_PASS0 << pass)) == 0 )
	{
            all_passes = 0;
	    break;
	}
    }
    if ( all_passes )
    {
       DIAG_PRINTF(1, ("    Failed all cal passes.\n"))
    }
    else if ( failure & E1432_CAL_FAIL_PASS_MASK )
    {
        DIAG_PRINTF(1, ("    Failed cal passes:\n"))
        for ( pass = 0; pass < passes; pass++ )
        {
	    if ( failure & (E1432_CAL_FAIL_PASS0 << pass) )
	    {
                DIAG_PRINTF(1, ("      %s\n", pass_str[pass]))
	    }
        }
    }
    if ( (failure & MEAS_PASS_INFO) && (sca_type == E1432_SCA_ID_SONATA_A
      || sca_type == E1432_SCA_ID_SONATA_B) )
    {
        if ( (failure & MEAS_PASS_INFO) == MEAS_PASS_INFO )
	{
            DIAG_PRINTF(1, (
	      "    Both low and high gain setting computations failed.\n"))
	}
	else if ( failure & E1432_CAL_FAIL_LOW_GAIN )
	{
            DIAG_PRINTF(1, ("    Low gain setting computations failed.\n"))
	}
	else /* E1432_CAL_FAIL_HIGH_GAIN only, by default */
	{
            DIAG_PRINTF(1, ("    High gain setting computations failed.\n"))
	}
    }
}

static void
cal_print_errors(LONGSIZ32 failure, int sca_type)
{
    char err_str[ERR_STRLEN + 1];
    DIAG_PRINTF(1, ("    %s.\n", cal_err_str(err_str, failure)))
    cal_print_passes(failure, sca_type);
}

static SHORTSIZ16
i1432_eval_cal_failure(SHORTSIZ16 la)
{
    SHORTSIZ16 err = 0;
    E1432ID hw = 0;
    struct e1432_hwconfig hwconfig;
    int chan, sca_chan, sca_base, chans, chans_per_sca;
    int sca, scas;
    int same;
    int sca_type;
    const char *const *sca_str;
    const char *const *chan_str;
    const SHORTSIZ16 *sca_fail;
    LONGSIZ32 chan_failures[E1432_INPUT_CHANS];
    LONGSIZ32 sca_failures[E1432_SCAS];
    LONGSIZ32 mod_failures;

    CHK(e1432_assign_channel_numbers(1, &la, &hw));
    CHK(e1432_get_hwconfig(1, &la, &hwconfig));

    sca_type = hwconfig.sca_id[0];
    chans = hwconfig.input_chans;
    switch( sca_type )
    {
    case E1432_SCA_ID_VIBRATO:
        chans_per_sca = 4;
	sca_str = e1432_in32_str;
	chan_str = e1432_chan32;
        sca_fail = e1432_fail32;
        break;
    case E1432_SCA_ID_SONATA_A:
        chans_per_sca = 2;
	sca_str = e1432_in33a_str;
	chan_str = e1432_chan33;
        sca_fail = e1432_fail33;
        break;
    case E1432_SCA_ID_SONATA_B:
        chans_per_sca = 2;
	sca_str = e1432_in33b_str;
	chan_str = e1432_chan33;
        sca_fail = e1432_fail33;
        break;
    default:
        DIAG_PRINTF(1, ("  Unknown sca found in first SCA slot.\n"))
	DIAG_PRINTF(0, ("  CAL test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }
    scas = chans/chans_per_sca;

    /* collect the error info */
    mod_failures = 0;
    chan = 0;
    for ( sca = 0; sca < scas; sca++ )
    {
	sca_failures[sca] = 0;
	for (sca_chan = 0; sca_chan < chans_per_sca; sca_chan++)
	{
            CHK(e1432_get_cal_failures(hw,
	      (SHORTSIZ16)E1432_INPUT_CHAN(chan+1), &chan_failures[chan]));
	    sca_failures[sca] |= chan_failures[chan];
	    mod_failures |= chan_failures[chan];
	    chan++;
	}
    }

    if ( mod_failures & FW_FAIL )
    {
        DIAG_PRINTF(1, ("  Cal firmware failure.\n"))
	DIAG_PRINTF(0, ("  CAL test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }
    if ( mod_failures & E1432_FLAG_CAL_FAILURE_NO_CONST )
    {
        DIAG_PRINTF(1, ("  Cal measurement found no subrate cal constants.\n"))
	DIAG_PRINTF(0, ("  CAL test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    }
    if ( mod_failures & E1432_FLAG_CAL_FAILURE_FIFO )
    {
        DIAG_PRINTF(1, ("  Cal measurement fifo failure.\n"))
        DIAG_PRINTF(1, ("  May be SCA bus hardware, substrate A Bus,"
	  " or substrate DMA & VALIDn path.\n"))
	DIAG_PRINTF(0, ("  CAL test failed.\n"));
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }
    if ( mod_failures & CAL_NUM_FAIL )
    {
	if ( scas > 1 )
	{
	    same = 1;
	    for ( chan = 1; chan < chans; chan++ )
            {
		if ( chan_failures[chan] != chan_failures[0] )
		{
		    same = 0;
		    break;
		}
	    }
	    if ( same && (chan_failures[0] & CAL_NUM_FAIL) )
	    {
                DIAG_PRINTF(1, ("  Cal measurement failed on all channels."))
	        DIAG_PRINTF(1, ("  (cal error word = 0x%8.8lx)\n",
		  chan_failures[0]))
		cal_print_errors(mod_failures, sca_type);
	        DIAG_PRINTF(0, ("  CAL test failed.\n"));
		if ( sca_type == E1432_SCA_ID_VIBRATO && scas == 1
		  || sca_type == E1432_SCA_ID_SONATA_A && scas == 2
		  || sca_type == E1432_SCA_ID_SONATA_B && scas == 2 )
		{
                    return DIAG_Q_ERR(ERR1432_DIAG_ERR_INPUT_SCA0);
		}
		else
		{
                    return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
		}
	    }
	    for ( sca = 0; sca < scas; sca++ )
	    {
		same = 1;
		sca_base = sca * chans_per_sca;
		for ( chan = sca_base + 1; chan < sca_base + chans_per_sca;
		  chan++ )
		{
		    if ( chan_failures[chan] != chan_failures[sca_base] )
		    {
		        same = 0;
		        break;
		    }
		}
	        if ( same && (chan_failures[sca_base] & CAL_NUM_FAIL) )
	        {
                    DIAG_PRINTF(1, ("  Cal measurement failed on"
		      " all channels of %s.\n", sca_str[sca]))
		    DIAG_PRINTF(1, ("  (cal error word = 0x%8.8lx)\n",
		      chan_failures[sca_base]))
		    cal_print_errors(mod_failures, sca_type);
		    if ( chan_failures[sca_base] & CAL_FAIL )
		    {
		        err = sca_fail[sca];
		    }
		    else /* chan_failures[sca_base] & NUM_FAIL by default */
		    {
		        err = ERR1432_DIAG_ERR_UNDIAGNOSED;
		    }
                    DIAG_Q_ERR(err);
	        }
	    }
	    if ( err )
	    {
	        DIAG_PRINTF(0, ("  CAL test failed.\n"));
		return err;
	    }
	}
	for ( chan = 0; chan < chans; chan++ )
	{
	    sca = chan / chans_per_sca;
	    if ( chan_failures[chan] & CAL_FAIL )
	    {
                DIAG_PRINTF(1, ("  Cal measurement failed on channel %d"
		  " (%s chan %s).\n", chan+1, sca_str[sca], chan_str[chan]))
	        DIAG_PRINTF(1, ("  (cal error word = 0x%8.8lx)\n",
		  chan_failures[chan]))
		cal_print_errors(mod_failures, sca_type);
                if ( err != sca_fail[sca] )
	        {
		    err = sca_fail[sca];
                    DIAG_Q_ERR(err);
	        }
	    }
        }
    }
    if ( err ) DIAG_PRINTF(0, ("  CAL test failed.\n"));
    return err;


cleanup: 
    DIAG_PRINTF(1, ("  Encountered unexpected error when evaluating"
      " cal failure:\n"))
    DIAG_PRINTF(1, ("    %d, %s\n", err, i1432_get_error_string(err)))
    DIAG_PRINTF(0, ("  CAL test failed.\n"));
    return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
}


/***************************************************************************
 Maps e1432 errors to PCA local errors, possibly prints diagnostic output.
 ***************************************************************************/
static SHORTSIZ16
i1432_map_errors(SHORTSIZ16 error, SHORTSIZ16 la, E1432_MODULE_LIST_NODE *mn)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 sca_fail, id;

    DIAG_PRINTF(2, ("  Mapping hostlib error number %d to board level.\n",
      error))

    if ( error >= 0 )
    {
        DIAG_PRINTF(0, ("  Undiagnosed failure, error %d\n", error))
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }

    /* reset page map, just in case */
    (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);

    switch(error)
    {
    case ERR1432_NO_PLL_LOCK:
    case ERR1432_BAD_CAL_BLOCK:
    case ERR1432_BAD_FLASH:
    case ERR1432_FLASH_INVALID:
        DIAG_PRINTF(1, ("  Substrate failure, error %d\n", error))
        DIAG_PRINTF(1, ("    %s\n", i1432_get_error_string(error)))
	DIAG_PRINTF(0, ("  Substrate failed.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
    case ERR1432_SCA_HOSTPORT_FAIL:
    case ERR1432_SCA_BOOT_FAIL:
    case ERR1432_SCA_BUS_ERROR:
    case ERR1432_SCA_FIRMWARE_ERROR:
    case ERR1432_SCA_EEPROM_INVALID:
	i1432_r32(mn, &err, (LONGSIZ32)E1432_DIAG_FAIL_SCA, &sca_fail);
        /* Currently these are only returned by Sonata. */
	/* Assuming a good sca_fail value returned, figure out whether an
	   A or B Sonata. */
	if ( sca_fail >= 0 || sca_fail <= 3 )
	{
	    i1432_r32(mn, &err, (LONGSIZ32)((int)E1432_SINFO_ID_REG
	      + sca_fail * E1432_SINFO_REG_OFFSET), &id);
	    if ( id == E1432_SCA_ID_SONATA_B )
	    {
                DIAG_PRINTF(1, ("  %s failure, error %d\n",
		  e1432_in33b_str[sca_fail], error))
	    }
	    else
	    {
                DIAG_PRINTF(1, ("  %s failure, error %d\n",
		  e1432_in33a_str[sca_fail], error))
	    }
	}
	else
	{
            DIAG_PRINTF(1, ("  Unknown SCA failure, error %d\n", error))
	}
	/* rest of error message */
        DIAG_PRINTF(1, ("    %s\n", i1432_get_error_string(error)))
	DIAG_PRINTF(0, ("  Download test failed.\n"));
	return DIAG_Q_ERR(e1432_fail33[sca_fail]);
    case ERR1432_CALIBRATION_FAILURE:
	return i1432_eval_cal_failure(la);
    case ERR1432_SELFTEST_FAILURE_SOURCE:
    case ERR1432_SOURCE_ERROR:
	i1432_r32(mn, &err, (LONGSIZ32)E1432_DIAG_FAIL_SCA, &sca_fail);
	/* convert scaindx to sca for ERR1432_SELFTEST_FAILURE_SOURCE */
	if ( error == ERR1432_SELFTEST_FAILURE_SOURCE ) sca_fail *= 2;
	DIAG_PRINTF(0, ("  Download test failed.\n"));
        DIAG_PRINTF(1, ("  %s failure, error %d\n",
	  e1432_src_str[sca_fail], error))
        DIAG_PRINTF(1, ("    %s\n", i1432_get_error_string(error)))
	return DIAG_Q_ERR(e1432_fail_src[sca_fail]);
    case ERR1432_DIAG_ERR_UNDIAGNOSED:
    case ERR1432_DIAG_ERR_SUBST:
    case ERR1432_DIAG_ERR_DRAM:
    case ERR1432_DIAG_ERR_TACH:
    case ERR1432_DIAG_ERR_SOURCE_FIT:
    case ERR1432_DIAG_ERR_SOURCE_SCA0:
    case ERR1432_DIAG_ERR_SOURCE_SCA1:
    case ERR1432_DIAG_ERR_INPUT_SCA0:
    case ERR1432_DIAG_ERR_INPUT_SCA1:
    case ERR1432_DIAG_ERR_INPUT_SCA2:
    case ERR1432_DIAG_ERR_INPUT_SCA3:
    default:
	/* already mapped and queued */
	return error;
    }
}

#if 0
/* Currently not used */
#ifdef  HAVE_SIGBUS
/*ARGSUSED*/
void
i1432_selftest_buserr_handler(LONGSIZ32 sig, LONGSIZ32 code,
  struct sigcontext *scp)
{
    /* return to previous context, signal = 1 */
    longjmp(i1432_buserr_env, 1);
}
#endif  /* HAVE_SIGBUS */
#endif


#if 0
/***************************************************************************
 Vibrato selftest.
 ***************************************************************************/
/*ARGSUSED*/
SHORTSIZ16
i1432_test_vibrato(SHORTSIZ16 la,
  E1432_MODULE_LIST_NODE *mn, E1432ID hw, LONGSIZ32 flags)
{
    SHORTSIZ16 err = 0;
    struct e1432_hwconfig hwconfig;
    LONGSIZ32 reg;
    SHORTSIZ16 sca = -1;
    int missing_sca = 0;
    int bus_err;
    SHORTSIZ16 page;
#ifdef  HAVE_SIGBUS
    struct sigvec vec;
    int save_state = i1432_buserr_trap;

    i1432_buserr_trap = 0;      /* Don't use normal set-try-recover */
#endif  /* HAVE_SIGBUS */

    DIAG_PRINTF(2, ("  i1432_test_vibrato(0x%.8x)", flags))

    if ( flags & E1432_SELFTEST_DOWNLOAD )
    {
#if 0   /* cal failure evaluation handled at time of download */
        DIAG_PRINTF(1, ("E1432 SCA downloaded test.\n"))
        CHK(e1432_get_hwconfig(1, &la, &hwconfig));
        if ( hwconfig.input_chans == 0)
        {
            DBPF(msg, (msg, "No input channels found!"))
            return NEEDS_ERRNUM;
        }
        /* should create a substrate variable, selftest, which allows a cal */
        err = i1432_dnld_fw_err;
        if ( err == ERR1432_CALIBRATION_FAILURE )
        {
            return i1432_eval_cal_failure(la);
        }
#endif
    }
    else
    {
#if 0  /* this shouldn't need all the bus err stuff, done much better in
	  hostdiag/testsca.c */
	LONGSIZ32 addr = E1432_SCA_ISR;

	page = E1432_SCA_PAGE;
	if (mn->a24_256k)
	    page <<= 2;
        CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, page));
	for ( sca = 0; sca < 4; sca++ )
	{
            bus_err = 0;

#ifdef  HAVE_SIGBUS
            vec.sv_handler = i1432_selftest_buserr_handler;
            vec.sv_mask = 0;
            vec.sv_flags = 0;
            (void) sigvector(SIGBUS, &vec, &vec);
            bus_err = setjmp(i1432_buserr_env);
#endif  /* HAVE_SIGBUS */

            if ( ! bus_err )
            {
	        /* DBPF(msg, (msg, "getting ready to do first read")) */
		i1432_r32(mn, &err, addr, &reg);
	    }

#ifdef  HAVE_SIGBUS
            /* Restore old handler and set-try-recover state */
            (void) sigvector(SIGBUS, &vec, &vec);
            i1432_buserr_trap = save_state;
#endif  /* HAVE_SIGBUS */

	    if ( err != ERR1432_BUS_ERROR ) break;
	    if ( bus_err || err )
	    {
                missing_sca = 1;
                DBPF(msg, (msg, "bus error on sca %d ISR @ addr 0x%.8x",
                  sca, addr))
		if ( flags & (E1432_SELFTEST_FORCE_SCA_1 << sca) )
		{
                    DBPF(msg, (msg, "sca %d fails to respond on the SCA bus",
                      (long)sca))
                    return NEEDS_ERRNUM;
		}
	    }
	    else
	    {
	        DBPF(msg, (msg, "sca %d ISR @ addr 0x%.8x = 0x%.8x",
	          (long)sca, addr, reg))
                if ( missing_sca )
		{
	            DBPF(msg, (msg, "WARNING, SCA slots improperly filled"
		      ", some tests will not locate SCA properly."))
		}
		/* SCA testing here */
		if ((reg & E1432_VIB_ISR_FIXED_BITS) !=
		    E1432_VIB_ISR_FIXED_VALS)
		{
	            DBPF(msg, (msg, "sca %d ISR bits in error", sca))
                    return NEEDS_ERRNUM;
		}
	    }
            addr += 0x400;
        }
#endif
    }

cleanup: 
    if ( err )
    {
        DIAG_PRINTF(2, ("  i1432_test_vibrato()returning %d", err))
    }
    return err;
}
#endif /* 0 */

#if 0
/***************************************************************************
 ***************************************************************************/
/*ARGSUSED*/
static SHORTSIZ16
i1432_read_fit_id_reg(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 * id_reg,
  LONGSIZ32 flags)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 sb, addr;

    if ( (err = i1432_map_sca(mn, E1432_SCA_FIT, &sb)) < 0 ) return err;
    sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */
    addr = sb + E1432_SRC_ID_REG;
    err = i1432_direct_ping_register(mn, addr);
    if ( ! err )
    {
	i1432_r32(mn, &err, sb + E1432_SRC_ID_REG, id_reg);
    }

/* FIX THIS!!!
    if ( err == ERR1432_BUS_ERROR && (flags & E1432_SELFTEST_FORCE_FIT))
    {
        DIAG_PRINTF(0, ("  FIT device fails to respond.\n"))
        return -1;
    }
*/

    return err;
}
#endif

#if 0
/***************************************************************************
 ***************************************************************************/
static SHORTSIZ16
i1432_clarinet_present(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 flags)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 id_reg = 0;

    err = i1432_read_fit_id_reg(mn , &id_reg, flags);
    if ( err == ERR1432_BUS_ERROR ) return 0;
    if ( err < 0 ) return err;
    if ( (id_reg & E1432_CLARI_ID_BITS) == E1432_CLARINET_ID ) return 1;
    else return 0;
}
#endif

static SHORTSIZ16
i1432_src_peek(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, SHORTSIZ16 sca,
  LONGSIZ32 memspace, LONGSIZ32 addr, LONGSIZ32 *word)
{
    SHORTSIZ16 err = 0;

    /* put into command buffer mode */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));

    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_WDREAD));
    CHK(i1432_write_src_hp_data(mn, sb, memspace));
    CHK(i1432_write_src_hp_data(mn, sb, addr << 8));

    /* terminate extended mesage */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));

    /* read and toss the first data word - trash */
    CHK(i1432_read_src_hp_data(mn, sb, word));
    /* read the data real */
    CHK(i1432_read_src_hp_data(mn, sb, word));

cleanup: 
    if ( err )
    {
        DIAG_PRINTF(1, ("  i1432_src_peek() error %d.\n", err))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}

static SHORTSIZ16
i1432_src_poke(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, SHORTSIZ16 sca,
  LONGSIZ32 memspace, LONGSIZ32 addr, LONGSIZ32 word)
{
    SHORTSIZ16 err = 0;

    DIAG_PRINTF(3, ("    i1432_src_poke(0x%lx, 0x%lx, %d,"
      " 0x%lx, 0x%lx, 0x%lx)\n", mn, sb, sca, memspace, addr, word))

    /* put into command buffer mode */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));

    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_WDWRITE));
    CHK(i1432_write_src_hp_data(mn, sb, memspace));
    CHK(i1432_write_src_hp_data(mn, sb, addr << 8));
    CHK(i1432_write_src_hp_data(mn, sb, word));

    /* terminate extended mesage */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));

cleanup: 
    if ( err )
    {
        DIAG_PRINTF(1, ("  i1432_src_poke() error %d\n", err))
        DIAG_PRINTF(2, ("    sca %d, memspace 0x%x, addr 0x%x, data 0x%x\n",
	  sca, memspace, addr, word))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}


static SHORTSIZ16
i1432_src_read_adc(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, LONGSIZ32 chan,
  LONGSIZ32 avgs, FLOATSIZ32* result_volts, FLOATSIZ32 settle_delay)
{
#define SCALE_FACTOR ((FLOATSIZ32)0x20000)
    SHORTSIZ16 err = 0;
    LONGSIZ32 reading;

/*
    DIAG_PRINTF(3, ("    i1432_src_read_adc(0x%x, %d, %d,"
      " %d, 0x%x, %.3f, %.3f)\n", mn, sb, chan,
		    avgs, result_volts, settle_delay));
*/
    
    /* put into command buffer mode */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONHPDCMDBUF));

    CHK(i1432_write_src_hp_data(mn, sb, E1432_CLARI_MONCMD_ADCREAD));
    CHK(i1432_write_src_hp_data(mn, sb, chan));
    CHK(i1432_write_src_hp_data(mn, sb, avgs << 8));

    /* Wait for Clarinet to (possibly) recover from going out of real-time
       from the previous activity */
    i1432_pause(settle_delay);

    /* terminate extended mesage */
    CHK(i1432_write_ack_src_hp_cmd(mn, sb, E1432_CLARI_MONEXTMSG));

    /* read and toss the first data word - trash */
    CHK(i1432_read_src_hp_data(mn, sb, &reading));
    /* read the data real */
    CHK(i1432_read_src_hp_data(mn, sb, &reading));

    /* shift down to lsbs */
    reading >>= 8;

    /* Apply the conversion factor to volts */
    *result_volts = (FLOATSIZ32)reading/SCALE_FACTOR;

cleanup: 
    if ( err )
    {
	DIAG_PRINTF(1, ("  Read/write to source ADC"
	  " through 56k hostport failed.\n"))
    }
    return err;
}


static SHORTSIZ16
i1432_src_adc_test(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb,
  SHORTSIZ16 sca, SHORTSIZ16 type)
{
    SHORTSIZ16 err = 0;
    FLOATSIZ32 adc_data;
    LONGSIZ32 chan;

    DIAG_PRINTF(2, ("  Testing source ADC(s).\n"))
    DIAG_PRINTF(3, ("    i1432_src_adc_test(0x%lx, 0x%lx, %d, %d)\n",
      mn, sb, sca, type))

    /* test ground input */
    if ( type == E1432_SCA_ID_CLARION )
    {
	chan = E1432_CLARION_ADC_CHAN_GND;
    }
    else
    {
	chan = E1432_CLARINET_ADC_CHAN_GND;
    }
    CHK(i1432_src_read_adc(mn, sb, chan, 1, &adc_data, 0.0f));
    /* ADC spec of +-6 LSB max bipolar offset works out to be 7 mV */
    if ( adc_data > .01 || adc_data < -.01 )
    {
        DIAG_PRINTF(1, ("  Source ADC GND data = %f,"
	  "expected 0 V +- 10 mV\n", adc_data))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }

    /* no reference channel on Clarion ADC */
    if ( type == E1432_SCA_ID_CLARION ) return 0;

    /* test 5V reference */
    chan = E1432_CLARINET_ADC_CHAN_VREF;
    CHK(i1432_src_read_adc(mn, sb, chan, 1, &adc_data, 0.0f));
    /* 5.0V nominal, use +- 10% as gross limit */
    if ( adc_data > 5.5 || adc_data < 4.5 )
    {
        DIAG_PRINTF(1, ("  Source ADC VREF data = %f,"
	  "expected 5 V +- .5 V\n", adc_data))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }

cleanup: 
    return err;
}


static SHORTSIZ16
i1432_clari_mem_test(E1432_MODULE_LIST_NODE *mn, LONGSIZ32 sb, SHORTSIZ16 sca)
{
#define ADDR_BITS 17  /* 16 address lines + memspace bit */
#define ADDR_MASK 0xffff  /* mask for the actual address bits */
#define MEM_MASK 0x10000  /* mask for the memspace bit in address word */
#define DATA_BITS 24
#define TEST_ADDR 0xa050
#define MEMSPACE(addr) \
  (addr & MEM_MASK ? E1432_CLARI_X_SPACE : E1432_CLARI_Y_SPACE)
    SHORTSIZ16 err = 0;
    int i;
    LONGSIZ32 read_word, write_word;
    LONGSIZ32 word_errs;
    LONGSIZ32 test_addr, altn_addr;

    DIAG_PRINTF(2, ("  Testing Source 56k RAM.\n"))
    DIAG_PRINTF(3, ("    i1432_clari_mem_test(0x%lx, 0x%lx)\n", mn, sb))

    /* marching "1" test of all data lines */
    word_errs = 0;
    test_addr = TEST_ADDR;
    altn_addr = ~test_addr;
    DIAG_PRINTF(3, ("    testing %d data lines\n", DATA_BITS))
    for ( i = 0; i < DATA_BITS; i++ )
    {
        write_word = 0x100 << i;
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_X_SPACE, test_addr,
	  write_word));
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_X_SPACE, altn_addr,
	  ~write_word));
        CHK(i1432_src_peek(mn, sb, sca, E1432_CLARI_X_SPACE, test_addr,
	  &read_word));
        word_errs |= read_word ^ write_word;
    }
    if ( word_errs )
    {
        DIAG_PRINTF(1, ("  Source RAM bit errors = 0x%x\n.", word_errs));
        DIAG_PRINTF(1, ("  Source test failed.\n"));
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }

    /* write unique data at addresses which differ in each address line */
    word_errs = 0;
    DIAG_PRINTF(3, ("    testing %d address lines\n", ADDR_BITS))
    for ( i = 0; i < ADDR_BITS; i++ )
    {
        /* flip one of the address bits */
        test_addr = TEST_ADDR^(1 << i);
        /* use the address for the data */
        write_word = test_addr << 8;
        CHK(i1432_src_poke(mn, sb, sca, MEMSPACE(test_addr),
	  test_addr & ADDR_MASK, write_word));
    }
    /* check for overwritten data */
    for ( i = 0; i < ADDR_BITS; i++ )
    {
        test_addr = TEST_ADDR^(1 << i);
        write_word = test_addr << 8;
        CHK(i1432_src_peek(mn, sb, sca, MEMSPACE(test_addr),
	  test_addr & ADDR_MASK, &read_word));
        DIAG_PRINTF(3, ("    read 0x%x at 0x%x, memspace 0x%x\n",
	  read_word, test_addr & ADDR_MASK, MEMSPACE(test_addr)))
        word_errs |= (read_word >> 8) ^ test_addr;
    }
    if ( word_errs & MEM_MASK )
    {
	DIAG_PRINTF(1, ("  Source RAM memory space selection failure.\n"))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    if ( word_errs & ADDR_MASK )
    {
        DIAG_PRINTF(1, ("  Source RAM address selection errors = 0x%x\n",
	  word_errs & ADDR_MASK))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }

cleanup: 
    if ( err )
    {
        DIAG_PRINTF(1, ("  Peek/poke to source RAM failed.  err = %d\n", err))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}


static SHORTSIZ16
i1432_src_freq_amp(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca, LONGSIZ32 src_romid,
 FLOATSIZ32 sine_freq, FLOATSIZ32 sine_phase, FLOATSIZ32 sine_range,
 FLOATSIZ32 sine_amp_scale)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 sb;
    E1432_MODULE_LIST_NODE *mn = NULL;

    DIAG_PRINTF(3, ("    i1432_src_freq_amp(0x%x, %d, %d,"
      " %.3f, %.3f, %.3f, %.3f)\n", hw, src_group, sca,
      sine_freq, sine_phase, sine_range, sine_amp_scale))

    CHK(e1432_set_range(hw, src_group, sine_range));
    CHK(e1432_set_amp_scale(hw, src_group, sine_amp_scale));
    CHK(e1432_set_sine_freq(hw, src_group, sine_freq));
    CHK(e1432_set_sine_phase(hw, src_group, sine_phase));

    /* to get pick up source parm changes */
    /* (shouldn't be necessary each time?) */
    CHK(e1432_init_measure(hw, src_group));
    i1432_pause(0.1);

    if (src_romid >= OVERREADTEST_SRCREV )
    {
	 CHK(i1432_get_module_from_chan(hw, src_group, &mn));
	 /* map in the source registers for peek/pokes */
	 /* this changes page map from base page */
	 CHK(i1432_map_sca(mn, sca, &sb));
	 sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */
	 CHK(i1432_write_nack_src_hp_cmd(mn, sb, E1432_CLARI_SINE2_NOOVERREAD));
	 i1432_pause(0.02);
     }
    
cleanup: 
    /* restore page map to base page */
    if (src_romid >= OVERREADTEST_SRCREV )
    {
	if ( mn ) (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    }
    if ( err )
    {
        DIAG_PRINTF(2, ("  i1432_src_freq_amp() returning %d", err))
    }
    return err;
}

#define MAX_OFFSET  .2  /* +- .2 V */
static SHORTSIZ16
i1432_test_src_dacs(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca, LONGSIZ32 src_romid)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 sb;
    FLOATSIZ32 mid, min, max;
    E1432_MODULE_LIST_NODE *mn = NULL;
    LONGSIZ32 adc_chan[2];
    int chan, chans;
    const char *const *chan_str;

    DIAG_PRINTF(2, ("  i1432_test_src_dacs(0x%lx, %d, %d 0x%x)\n",
      hw, src_group, sca, src_romid))

    CHK(i1432_get_module_from_chan(hw, src_group, &mn));

    if ( sca == E1432_SCA_FIT )
    /* Clarinet */
    {
	chans = 1;
	adc_chan[0] = E1432_CLARINET_ADC_CHAN_AMPOUT;
        chan_str = e1432_no_chan;
    }
    else
    /* Clarion */
    {
	chans = 2;
	adc_chan[0] = E1432_CLARION_ADC_CHAN_AMPOUT1;
	adc_chan[1] = E1432_CLARION_ADC_CHAN_AMPOUT2;
        chan_str = e1432_src_chan;
    }

    /* set range and start default signal generation mode, sine, to output 0 Volts) */
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, 0, 0, 10.0f, 0.0f));

    /* map in the source registers for peek/pokes */
    /* this changes page map from base page */
    CHK(i1432_map_sca(mn, sca, &sb));
    sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */

    /* both DACs to midscale, measure residual offset */
    for ( chan = 0; chan < chans; chan++ )
    {
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, pre_dac[chan],
          0x80000));
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, post_dac[chan],
          0x80000));
        i1432_pause(0.1);
        CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &mid,
	  E1432_CLARI_ADC_SETTLE));
    
        if ( mid < -MAX_OFFSET || mid > MAX_OFFSET )
        {
            DIAG_PRINTF(1, ("  %s%s residual offset (not autozero'd)"
	      " failed.\1", e1432_src_str[sca], chan_str[chan]))
            DIAG_PRINTF(1, ("    output = %f, should be between %f and %f\n",
	      mid, -MAX_OFFSET, MAX_OFFSET))
            DIAG_PRINTF(0, ("  Source test failed.\n"));
	    return DIAG_Q_ERR(e1432_fail_src[sca]);
        }
    }

    /* pre rangedac max */
    for ( chan = 0; chan < chans; chan++ )
    {
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, pre_dac[chan],
          0x00000));
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, post_dac[chan],
          0x80000));
        i1432_pause(0.1);
        CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &max,
	  E1432_CLARI_ADC_SETTLE));
    
        /* pre rangedac min */
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, pre_dac[chan],
          0xfff00));
        i1432_pause(0.1);
        CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &min,
	  E1432_CLARI_ADC_SETTLE));
    
        if ( max-min < .4 || max-min > .6 )
        {
            DIAG_PRINTF(1, ("  %s%s DC Offset DAC pin 28 out failed.\n",
	      e1432_src_str[sca], chan_str[chan]))
            DIAG_PRINTF(1, ("    output, min = %f, max = %f,"
	      " max-min should be > .4, < .6\n", max, min))
            DIAG_PRINTF(0, ("  Source test failed.\n"));
	    return DIAG_Q_ERR(e1432_fail_src[sca]);
        }
    }

    /* post rangedac max */
    for ( chan = 0; chan < chans; chan++ )
    {
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, pre_dac[chan],
          0x80000));
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, post_dac[chan],
          0x00000));
        i1432_pause(0.1);
        CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &max,
	  E1432_CLARI_ADC_SETTLE));
    
        /* post rangedac min */
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, post_dac[chan],
          0xfff00));
        i1432_pause(0.1);
        CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &min,
	  E1432_CLARI_ADC_SETTLE));
    
        if ( max-min < .35 || max-min > .55 )
        {
            DIAG_PRINTF(1, ("  %s%s DC Offset DAC pin 3 out failed.\n",
	      e1432_src_str[sca], chan_str[chan]))
            DIAG_PRINTF(1, ("    output, min = %f, max = %f,"
	      " max-min should be > .35, < .55\n", max, min))
            DIAG_PRINTF(0, ("  Source test failed.\n"));
	    return DIAG_Q_ERR(e1432_fail_src[sca]);
        }
    }

    /* restore both DACs to midscale */
    for ( chan = 0; chan < chans; chan++ )
    {
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, pre_dac[chan],
          0x80000));
        CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, post_dac[chan],
          0x80000));
    }

cleanup: 
    /* restore page map to base page */
    if ( mn ) (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    if ( err )
    {
        DIAG_PRINTF(1, ("  Source DAC test failed with err = %d.\n", err))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
	return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}

static SHORTSIZ16
i1432_test_src_out(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca, LONGSIZ32 src_romid,
  FLOATSIZ32 sine_freq, FLOATSIZ32 sine_amp, FLOATSIZ32 max_offset,
  FLOATSIZ32 pos_gain_err, FLOATSIZ32 neg_gain_err)
{
#define AC_LOOPS   50   /* number of samples if an AC output */
    SHORTSIZ16 err = 0;
    LONGSIZ32 sb;
    E1432_MODULE_LIST_NODE *mn = 0;
    LONGSIZ32 adc_chan[2];
    int chan, chans;
    const char *const *chan_str;
    FLOATSIZ32 adc_data, min_adc_data, max_adc_data, pk_to_pk_adc_data;
    FLOATSIZ32 min, max, min_pk_to_pk, max_pk_to_pk;
    int i, j;
    FLOATSIZ32 dbf[AC_LOOPS];
    FLOATSIZ32 tmp_data;
    FLOATSIZ32 adc_settle = E1432_CLARI_ADC_SETTLE;

    DIAG_PRINTF(3, ("    i1432_test_src_out(0x%lx, %d, %d, 0x%lx,"
      " %.2f, %.3f, %.3f, %.3f, %.3f)\n",
      hw, src_group, sca, src_romid,
      sine_freq, sine_amp, max_offset, pos_gain_err, neg_gain_err))

    CHK(i1432_get_module_from_chan(hw, src_group, &mn));

    if ( sca == E1432_SCA_FIT )
    /* Clarinet */
    {
	chans = 1;
	adc_chan[0] = E1432_CLARINET_ADC_CHAN_AMPOUT;
        chan_str = e1432_no_chan;
    }
    else
    /* Clarion */
    {
	chans = chans_in_group(hw, src_group);
	adc_chan[0] = E1432_CLARION_ADC_CHAN_AMPOUT1;
	adc_chan[1] = E1432_CLARION_ADC_CHAN_AMPOUT2;
        chan_str = e1432_src_chan;
    }

    /* map in the source registers for peek/pokes */
    /* this changes page map from base page */
    CHK(i1432_map_sca(mn, sca, &sb));
    sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */

    if ( sine_freq == 0.0 )
    {
        if ( sine_amp >= 0 ) 
	{
            max = (FLOATSIZ32)(sine_amp*(1.0 + pos_gain_err) + max_offset);
            min = (FLOATSIZ32)(sine_amp*(1.0 + neg_gain_err) - max_offset);
	}
	else
	{
            max = (FLOATSIZ32)(sine_amp*(1.0 + neg_gain_err) + max_offset);
            min = (FLOATSIZ32)(sine_amp*(1.0 + pos_gain_err) - max_offset);
	}
    }
    else
    {
        max = (FLOATSIZ32)(sine_amp*(1.0 + pos_gain_err) + max_offset);
	min = -max;
	max_pk_to_pk = (FLOATSIZ32)(2.0*sine_amp*(1.0 + pos_gain_err));
	min_pk_to_pk = (FLOATSIZ32)(2.0*sine_amp*(1.0 + neg_gain_err));
    }

    if ( sine_freq == 0.0 )
    {
        for ( chan = 0; chan < chans; chan++ )
        {
            CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &adc_data,
	      adc_settle));
            if ( adc_data > max || adc_data < min )
            {
                DIAG_PRINTF(1, ("  %s%s signal test failure.\n",
	          e1432_src_str[sca], chan_str[chan]))
                DIAG_PRINTF(1, ("    Received %.3f, expected %.3f to %.3f\n",
	          adc_data, min, max))
	        return e1432_fail_src[sca];
	    }
        }
    }
    else
    {
        for ( chan = 0; chan < chans; chan++ )
        {
            min_adc_data = max;
	    max_adc_data = min;
	    /* wait long enough to get at least 1 full cycle total */
	    if ( 1.0/sine_freq/(FLOATSIZ32)AC_LOOPS > E1432_CLARI_ADC_SETTLE )
	    {
	        adc_settle = 1.0f/sine_freq/(FLOATSIZ32)AC_LOOPS;
	    }
	    for ( i = 0; i < AC_LOOPS; i++ )
	    {
                CHK(i1432_src_read_adc(mn, sb, adc_chan[chan], 1, &adc_data,
		  adc_settle));
                dbf[i] = adc_data;
                for ( j = i; j > 0; j-- )
	        {
		    if ( dbf[j-1] > dbf[j] )  
		    {
                        tmp_data = dbf[j-1];
		        dbf[j-1] = dbf[j];
		        dbf[j] = tmp_data;
		    }
	        }
	    }
	    max_adc_data = dbf[AC_LOOPS-1];
	    min_adc_data = dbf[0];
            if ( max_adc_data > max || min_adc_data < min )
            {
                DIAG_PRINTF(1, ("  %s%s signal test failure.\n",
	          e1432_src_str[sca], chan_str[chan]))
                DIAG_PRINTF(1, ("    Received signal values from"
		  " %.3f to %.3f.\n", min_adc_data, max_adc_data))
	        DIAG_PRINTF(1, ("    Expected to be within %.3f to %.3f.\n",
	          min, max))
	        return e1432_fail_src[sca];
            }
	    pk_to_pk_adc_data = max_adc_data - min_adc_data;
            if ( pk_to_pk_adc_data < min_pk_to_pk 
	      || pk_to_pk_adc_data > max_pk_to_pk )
            {
                DIAG_PRINTF(1, ("  %s%s signal test failure.\n",
	          e1432_src_str[sca], chan_str[chan]))
                DIAG_PRINTF(1, ("    Received %.3f peak to peak.\n",
	          pk_to_pk_adc_data))
	        DIAG_PRINTF(1, ("    Expected to be within %.3f to %.3f p/p.\n",
	          min_pk_to_pk, max_pk_to_pk))
	        return e1432_fail_src[sca];
            }
	}
    }

cleanup: 
    /* restore page map to base page */
    if ( mn ) (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    if ( err )
    {
        DIAG_PRINTF(2, ("  i1432_test_src_out() returning %d", err))
    }
    return err;
}


/***************************************************************************
 ***************************************************************************/
static SHORTSIZ16
i1432_test_src_sigpath(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca,
		       LONGSIZ32 src_romid)
{
    SHORTSIZ16 err = 0;
    FLOATSIZ32 freq, phase, range, scale;

    DIAG_PRINTF(2, ("  i1432_test_src_sigpath(0x%lx, %d, %d, 0x%lx)\n",
      hw, src_group, sca, src_romid))
 
    CHK(e1432_set_filter_freq(hw, src_group, 25600));
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));
    /* CHK(e1432_auto_zero(hw, src_group)); */
    range = 10.0f;
    scale = .5f;

    freq = 0.f;
    phase = 90.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    phase = -90.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, -range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    freq = 1000.f;
    phase = 0.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    /* should still be flat at 20 kHz */
    freq = 20000.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

cleanup: 
    if ( err )
    {
        if ( err == ERR1432_DIAG_ERR_SOURCE_FIT
          || err == ERR1432_DIAG_ERR_SOURCE_SCA0
          || err == ERR1432_DIAG_ERR_SOURCE_SCA1 )
	{
	    DIAG_PRINTF(1, ("    DF1700, 25.6k filt signal path\n"))
	}
	else
	{
            DIAG_PRINTF(1, ("  Source DF1700, 25.6k filt signal path"
	      " test failure %d.\n", err))
	}
        DIAG_PRINTF(1, ("    %.3f Hz, %.3f degrees, %.3f range, %.3f scale.\n",
          freq, phase, range, scale))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
        return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}


/***************************************************************************
 Checks out the non DF1700 path and the 25.6 kHz reconstruction filter.
 ***************************************************************************/
static SHORTSIZ16
i1432_test_src_alt_sigpath(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca,
			   LONGSIZ32 src_romid)
{
    SHORTSIZ16 err = 0;
    FLOATSIZ32 freq, phase, range, scale;

    DIAG_PRINTF(2, ("  i1432_test_src_alt_sigpath("
      "0x%lx, %d, %d)\n", hw, src_group, sca))

    range = 10.f;
    scale = .5f;
    phase = 0.f;

    /* CHK(e1432_auto_zero(hw, src_group)); */
    CHK(e1432_set_filter_freq(hw, src_group, 6400));
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_OFF));

    freq = 1000.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    /* +.25 dB at band edge */
    freq = 6400.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, 1.03f*range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    /* must turn filters on to get all the way up to this frequency */
    /* -77 dB @ 34.56 kHz, 54 dB/8ave -> -34.2 dB @ 20kHz */
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));
    freq = 20000.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, .0195f*range*scale,
      OFFSET_LIMIT, TRANSBAND_AMP_LIMIT_POS, TRANSBAND_AMP_LIMIT_NEG));

    /* restore to normal */
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));
    CHK(e1432_set_filter_freq(hw, src_group, 25600));
    freq = 1000.f;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));

cleanup: 
    if ( err )
    {
        if ( err == ERR1432_DIAG_ERR_SOURCE_FIT
          || err == ERR1432_DIAG_ERR_SOURCE_SCA0
          || err == ERR1432_DIAG_ERR_SOURCE_SCA1 )
	{
	    DIAG_PRINTF(1, ("    non-DF1700, 6.4k filt signal path\n"))
	}
	else
	{
            DIAG_PRINTF(1, ("  Source non-DF1700, 6.4k filt signal path"
	      " test failure %d.\n", err))
	}
        DIAG_PRINTF(1, ("    %.3f Hz, %.3f range, %.3f scale.\n",
	  freq, range, scale))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
        return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}


/***************************************************************************
 Tests Clarinet shutdown directly, by poking the two shutdown addresses.
 ***************************************************************************/
static SHORTSIZ16
i1432_test_src_shutdown(E1432ID hw, SHORTSIZ16 src_group, SHORTSIZ16 sca,
			LONGSIZ32 src_romid)
{
    SHORTSIZ16 err = 0;
    E1432_MODULE_LIST_NODE *mn;
    FLOATSIZ32 freq, phase, range, scale;
    char not_shut[] = "not shut down";
    char slow_shut[] = "after slow shut down";
    /*char fast_shut[] = "after fast shut down";*/
    char *shut;
    LONGSIZ32 sb;

    DIAG_PRINTF(2, ("  i1432_test_src_shutdown("
      "0x%lx, %d, %d)\n", hw, src_group, sca))

    CHK(i1432_get_module_from_chan(hw, src_group, &mn));

    freq = 3000.f;
    phase = 0.f;
    range = 10.f;
    scale = .5f;

    CHK(e1432_set_filter_freq(hw, src_group, 25600));
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));

    /* put out 5V p/p and check for it */
    shut = not_shut;
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, freq, phase, range, scale));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -INBAND_AMP_LIMIT));

    /* start slow shutdown and check */
    shut = slow_shut;
    /* map in the source registers for poke */
    /* this changes page map from base page */
    CHK(i1432_map_sca(mn, sca, &sb));
    sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */
    CHK(i1432_src_poke(mn, sb, sca, E1432_CLARI_Y_SPACE, E1432_CLARI_SSHUT, 0));
    /* restore the page map to base page */
    CHK(i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0));
    /* would be a slow rampdown but asserts the USERSDn which causes FW to
       shut down the DAC immediately */
    i1432_pause(.1);
    /* first, dummy test flushes out some un-settled measurements that
       appear only on the VXLink based tests (production) */
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, range*scale,
      OFFSET_LIMIT, INBAND_AMP_LIMIT, -1.0f));
    CHK(i1432_test_src_out(hw, src_group, sca, src_romid, freq, OFF_LIMIT,
      OFFSET_LIMIT, 0.0f, -1.0f));

    /* The fast shutdown was moved over to the board RESETn line so can
       no longer be tested. */

cleanup: 
    /* restore page map to base page */
    if ( mn ) (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    if ( err )
    {
        if ( err == ERR1432_DIAG_ERR_SOURCE_FIT
          || err == ERR1432_DIAG_ERR_SOURCE_SCA0
          || err == ERR1432_DIAG_ERR_SOURCE_SCA1 )
	{
	    DIAG_PRINTF(1, ("    Shutdown test, %s\n", shut))
	}
	else
	{
            DIAG_PRINTF(1, ("  Shutdown test failure %d with %s.\n",
	      err, shut))
	}
        DIAG_PRINTF(1, ("    %.3f Hz, %.3f range, %.3f scale.\n",
          freq, range, scale))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
        return DIAG_Q_ERR(e1432_fail_src[sca]);
    }
    return err;
}


/***************************************************************************
 Clarinet selftest, no download.

 Only deals with one module and only one SCA at a time.
 ***************************************************************************/
/*ARGSUSED*/
static SHORTSIZ16
i1432_test_src(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 sca, SHORTSIZ16 type)
{
    SHORTSIZ16 err = 0;
    LONGSIZ32 sb;	/* source base address, returned by map_src function */


    DIAG_PRINTF(1, ("Testing %s.\n", e1432_src_str[sca]))
    DIAG_PRINTF(2, ("  i1432_test_src(0x%lx, %d, %d)\n", mn, sca, type))

    /* set up page map reg, return sca/fit base address in sb */        
    CHK(i1432_map_sca(mn, sca, &sb));
    sb &= BOTTOM_1M_WIN; /* kludge to put in 1M fixed window */

    /* checks for correct board ID, echo back from hostport */
    err = i1432_src_ok(mn, sb);
    /* all failures are source failures */
    if ( err )
    {
        DIAG_PRINTF(0, ("  Source test failed.\n"));
        err = DIAG_Q_ERR(e1432_fail_src[sca]);
        goto cleanup;
    }

    CHK(i1432_clari_mem_test(mn, sb, sca));

    CHK(i1432_src_adc_test(mn, sb, sca, type));

cleanup: 
    if ( err )
    {
	DIAG_PRINTF(2,("  i1432_test_src() failed with error %d.\n", err))
    }
    return err;
}


/***************************************************************************
 Clarinet selftest, version with downloaded code.

 Only deals with one module.
 ***************************************************************************/
/*ARGSUSED*/
static SHORTSIZ16
i1432_test_src_dld(E1432ID hw, SHORTSIZ16 sca, int src_chans)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 src_list[2];
    SHORTSIZ16 src_group;
    /* SHORTSIZ16 ch1_group; Clariion bug? */
    SHORTSIZ16 src_n;
    LONGSIZ32 romdate;
    LONGSIZ32 bdid;
    char bddate[13];
    LONGSIZ32 src_romid = 0;

    DIAG_PRINTF(1, ("Testing %s.\n", e1432_src_str[sca]))
    DIAG_PRINTF(2, ("  i1432_test_src_dld(0x%lx, %d, 0x%lx, %d)\n",
      hw, sca, src_romid, src_chans))

    /* Create channel group */
    if ( sca == E1432_SCA_FIT )
    {
        src_n = 1;
        src_list[0] = E1432_SOURCE_CHAN(src_chans);
    }
    else
    {
        src_n = 2;
        src_list[0] = E1432_SOURCE_CHAN(sca + 1);
        src_list[1] = E1432_SOURCE_CHAN(sca + 2);
    }
    src_group = e1432_create_channel_group(hw, src_n, src_list);
    /* Clarion bug with doing a single channel group?
    ch1_group = e1432_create_channel_group(hw, 1, src_list);
    */
    if ( src_group >= 0 /* || ch1_group >= 0 Clarion bug? */ )
    {
	DIAG_PRINTF(1,
	  ("  e1432_create_channel_group() failed with error %d.\n", src_group))
        DIAG_PRINTF(0, ("  Source test failed.\n"));
        err = DIAG_Q_ERR(e1432_fail_src[sca]);
	goto cleanup;
    }

    CHK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

    /* get source rom id for that sca */
    CHK(e1432_set_srcbuffer_init(hw, src_group,E1432_SRCBUFFER_INIT_XFER));
    CHK(e1432_src_get_rev(hw, src_list[0], &src_romid, &romdate, &bdid, (char *)bddate));

    /*
    CHK(e1432_set_filter_freq(hw, src_group, 6400));
    */
    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));
    CHK(e1432_set_source_cola(hw, src_group, E1432_SOURCE_COLA_OFF));
    CHK(e1432_set_source_sum(hw, src_group, E1432_SOURCE_SUM_OFF));
    CHK(e1432_set_ramp_rate(hw, src_group, 0));

    /* Turn off the source polling in the 96k background - interferes
       with the peeks/pokes used by these tests. */
    CHK(e1432_write32_register(hw, src_list[0], (LONGSIZ32)E1432_DIAG_MODES,
      (LONGSIZ32)E1432_DIAG_MODE_NO_SRC_POLL));

    CHK(i1432_test_src_dacs(hw, src_group, sca, src_romid));
    CHK(i1432_test_src_sigpath(hw, src_group, sca, src_romid));
    if ( sca == E1432_SCA_FIT ) /* bug in Clarion? */
    {
        /* only chan 1 has alternate signal path */
        /* CHK(i1432_test_src_alt_sigpath(hw, ch1_group, sca));
	Clarion bug? */ CHK(i1432_test_src_alt_sigpath(hw, src_group, sca, src_romid));
    }
    CHK(i1432_test_src_shutdown(hw, src_group, sca, src_romid));

    /* clear page map, special modes */
    (void)e1432_write_register(hw, src_list[0], E1432_PAGE_MAP_REG, 0);
    (void)e1432_write32_register(hw, src_list[0], (LONGSIZ32)E1432_DIAG_MODES,
      (LONGSIZ32) 0 );

    CHK(e1432_reset_measure(hw, src_group));

#if 0  /* FIX!!! E1432 error -1409 */
    CHK(e1432_set_active(hw, src_group, E1432_CHANNEL_OFF));
#endif

cleanup: 
    if ( err )
    {
	DIAG_PRINTF(2,("  i1432_test_src_dld() failed with error %d.\n",
	  err))
    }
    return err;
}


/***************************************************************************
 Turn on Clarinet/Clarion output with  1 kHz, 1 Vp/p output.
 Done last so that output is there for user to verify.

 Only deals with one module.
 ***************************************************************************/
static SHORTSIZ16
i1432_test_src_std(E1432ID hw, SHORTSIZ16 sca, int src_chans)
{
    SHORTSIZ16 err = 0;
    SHORTSIZ16 src_list[2];
    SHORTSIZ16 src_group;
    SHORTSIZ16 src_n;
    LONGSIZ32 romdate;
    LONGSIZ32 bdid;
    char bddate[13];
    LONGSIZ32 src_romid = 0;

    DIAG_PRINTF(2, ("  i1432_test_src_std(0x%lx, %d, %d)\n",
      hw, sca, src_chans))

    /* Create channel group */
    if ( sca == E1432_SCA_FIT )
    {
        src_n = 1;
        src_list[0] = E1432_SOURCE_CHAN(src_chans);
    }
    else
    {
        src_n = 2;
        src_list[0] = E1432_SOURCE_CHAN(sca + 1);
        src_list[1] = E1432_SOURCE_CHAN(sca + 2);
    }
    src_group = e1432_create_channel_group(hw, src_n, src_list);
    if ( src_group >= 0 )
    {
        /* try a second time, to flush the error 1409 */
	src_group = e1432_create_channel_group(hw, src_n, src_list);
	if ( src_group >= 0 )
	{
	    DIAG_PRINTF(1, ("  e1432_create_channel_group() failed"
	      " with error %d.\n", src_group))
	    DIAG_PRINTF(0, ("  Source test failed.\n"));
	    err = DIAG_Q_ERR(e1432_fail_src[sca]);
	    goto cleanup;
	}
    }

    CHK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));

    /* get source rom id for that sca */
    CHK(e1432_set_srcbuffer_init(hw, src_group,E1432_SRCBUFFER_INIT_XFER));
    CHK(e1432_src_get_rev(hw, src_list[0], &src_romid, &romdate, &bdid, (char *)bddate));

    CHK(e1432_set_anti_alias_digital(hw, src_group,
      E1432_ANTI_ALIAS_DIGITAL_ON));
    CHK(e1432_set_source_sum(hw, src_group, E1432_SOURCE_SUM_OFF));
    CHK(e1432_set_ramp_rate(hw, src_group, 0));

    /* manual verification of 1 kHz, 1 Vp/p output and COLA ON */
    CHK(e1432_set_source_cola(hw, src_group, E1432_SOURCE_COLA_ON));
    CHK(e1432_set_filter_freq(hw, src_group, 25600));
    CHK(i1432_src_freq_amp(hw, src_group, sca, src_romid, 1000, 0, 2.0f, .5f));

cleanup: 
    if ( err )
    {
	DIAG_PRINTF(2,("  i1432_test_src_std() failed with error %d.\n",
	  err))
    }
    return err;
}


#define MAX_ERRS	9	/* really MAX_ERRS - 1 */
#define MAX_CHANS	16
#define BLKSIZE		256L
#define XPECTED_PK	1	/* 1 VPk expected at inputs */
#define S32		E1432_SCA_ID_VIBRATO	/* shorthand */
#define AC		E1432_COUPLING_AC	/* shorthand */
#define DC		E1432_COUPLING_DC	/* shorthand */
#define MAX_OVERHEAD	5.0f	/* above range before clipping */
#define MAX_DC_OFFSET		.1f	/* maximum, in V, on low DC ranges */
#define MAX_AC_OFFSET		.01f	/* maximum, in V, on low AC ranges */
#define MAX_DC_OFFSET_FACTOR	.1f	/* fraction of range, maximum */
#define GAIN_MIN		.8f	/* minimum gain relative to 1.0 */
#define GAIN_MAX		1.2f	/* maximum gain relative to 1.0 */
static const char couple_str_ac[] = "AC";
static const char couple_str_dc[] = "DC";
static SHORTSIZ16
i1432_test_in_std(E1432ID hw, struct e1432_hwconfig hwconfig, SHORTSIZ16 *scas)
{
    SHORTSIZ16 err = 0, errs = 0;
    SHORTSIZ16 nchan = hwconfig.input_chans;
    SHORTSIZ16 chans = 0;
    SHORTSIZ16 group, status, sca;
    const SHORTSIZ16 *chan2sca, *sca_err;
    SHORTSIZ16 chan_list[MAX_CHANS];
    SHORTSIZ16 tcount, i, j;
    FLOATSIZ32 buffer[BLKSIZE];
    LONGSIZ32 count;
    FLOATSIZ32 range, sqr_range;
    SHORTSIZ16 coupling;
    FLOATSIZ32 neg_peak, pos_peak, peak, offset, avg_sqr, tmp;
    FLOATSIZ32 avg_sqr_min, avg_sqr_max, peak_min, peak_max, offset_max;
    SHORTSIZ16 sca_type = E1432_SCA_ID_NONE;
    FLOATSIZ32 range_tbl[] =
                               { 1.0f, 2.0f, 5.0f, 10.0f, .5f, .2f, .1f, .05f };
    SHORTSIZ16 coupling_tbl[] =
                               {   DC,   DC,   DC,    DC,  AC, AC,  AC,    AC };
    SHORTSIZ16 not_do_tbl[] =
                               {    0,    0,    0,   S32, S32,  0,   0,   S32 };
    SHORTSIZ16 loops = sizeof(range_tbl)/sizeof(FLOATSIZ32);
    SHORTSIZ16 loop = 0;
    SHORTSIZ16 fail_sca[] = { 0, 0, 0, 0, 0 };
    const char *couple_str;

    for ( sca = 0; sca <  E1432_SCAS; sca++ )
    {
        switch( scas[sca] )
        {
        case E1432_SCA_ID_VIBRATO:
	    chans += 4;
	    sca_type = scas[sca];
	    chan2sca = e1432_32chan2sca;
	    sca_err = e1432_fail32;
	    break;
        case E1432_SCA_ID_SONATA_A:
        case E1432_SCA_ID_SONATA_B:
	    chans += 2;  /* 2 per SCA connector, 4 per SCA board */
	    sca_type = scas[sca];
	    chan2sca = e1432_33chan2sca;
	    sca_err = e1432_fail33;
	    break;
	}
    }

    if ( chans == 0 ) return err;

    DIAG_PRINTF(1, ("Testing inputs assuming 1 VPk, 1 KHz input.\n"))

    for ( i = 0, j = 0; i < nchan; i++ )
    {
	if ( scas[chan2sca[i]] == sca_type )
	{
	    chan_list[j++] = E1432_INPUT_CHAN(i+1);
	}
    }

    if ( nchan < 4 || j != chans )
    {
        DIAG_PRINTF(1, ("  Module channel count mismatch error.\n"))
        DIAG_PRINTF(0, ("  Input std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }

    group = e1432_create_channel_group(hw, chans, chan_list);
    if (group >= 0)
    {
	DIAG_PRINTF(1, ("  e1432_create_channel_group returned %d\n", group))
	DIAG_PRINTF(0, ("  Input std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }

    CHK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32));
    CHK(e1432_set_blocksize(hw, group, BLKSIZE));
    CHK(e1432_set_clock_freq(hw, group, 51200.0f));

    while ( loop < loops )
    {
	range = range_tbl[loop];
	sqr_range = range * range;
	coupling = coupling_tbl[loop];
        CHK(e1432_set_range(hw, group, range));
        CHK(e1432_set_coupling(hw, group, coupling));
        couple_str = ( coupling == E1432_COUPLING_AC )
	  ?  couple_str_ac : couple_str_dc;

        /* Start measurement */
        CHK(e1432_init_measure(hw, group));

	/* Wait for block available */
	tcount = 1000;
	while ( (status = e1432_block_available(hw, group)) == 0
	  && tcount-- > 0 );
	if (status <= 0)
	{
	    DIAG_PRINTF(1, ("  Error %d from e1432_block_available.\n", status))
	    DIAG_PRINTF(0, ("  Input std in test failed.\n"))
            return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
	}

	/* Read data */
	for (i = 0; i < chans; i++)
	{
	    CHK(e1432_read_float32_data(hw, chan_list[i],
					  E1432_TIME_DATA, buffer,
					  BLKSIZE, NULL, &count));
	    if (count != BLKSIZE)
	    {
	        DIAG_PRINTF(1, ("  Data count error, expected %ld,"
		  " received %ld.\n", BLKSIZE, count))
	        DIAG_PRINTF(0, ("  Input std in test failed.\n"))
                return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
	    }
	    /* check the time data received */
            neg_peak = 1.E6f;
	    pos_peak = -1.E6f;
	    avg_sqr = 0.0f;
	    offset = 0.0f;
	    for ( j = 0; j < BLKSIZE; j++ )
	    {
		tmp = buffer[j];
		if ( tmp < neg_peak ) neg_peak = tmp;
		else if ( tmp > pos_peak ) pos_peak = tmp;
		avg_sqr += tmp*tmp;
		offset += tmp;
	    }
	    avg_sqr /= (float) BLKSIZE;
	    offset /= (float) BLKSIZE;
            peak = .5f * (pos_peak - neg_peak);

            peak_min = GAIN_MIN * XPECTED_PK;
	    peak_max = GAIN_MAX * XPECTED_PK;
	    avg_sqr_min = .5f * peak_min*peak_min;
	    avg_sqr_max = .5f * peak_max*peak_max;
	    if ( peak_min > range ) peak_min = range;
	    if ( peak_max > MAX_OVERHEAD * range )
	    {
		peak_max = MAX_OVERHEAD * range;
	    }
	    if ( avg_sqr_min > sqr_range ) avg_sqr_min = sqr_range;
	    if ( avg_sqr_max > MAX_OVERHEAD * MAX_OVERHEAD * sqr_range )
	    {
	        avg_sqr_max = MAX_OVERHEAD * MAX_OVERHEAD * sqr_range;
	    }
            if ( coupling == E1432_COUPLING_AC )
	    {
                if ( XPECTED_PK <= range )
		{
		    offset_max = MAX_DC_OFFSET_FACTOR * range;
		    if ( offset_max < MAX_AC_OFFSET )
		    {
			offset_max = MAX_AC_OFFSET;
		    }
		}
		else
		{
		    offset_max = peak_max;
		}
	    }
	    else
	    {
		offset_max = MAX_DC_OFFSET_FACTOR * range;
		if ( offset_max < MAX_DC_OFFSET ) offset_max = MAX_DC_OFFSET;
	    }
	    if ( avg_sqr > avg_sqr_max || avg_sqr < avg_sqr_min
	      || peak > peak_max || peak < peak_min
	      || offset > offset_max || offset < -offset_max )
            {
		if ( ++errs < MAX_ERRS )
		{
	            DIAG_PRINTF(1, ("  On Channel %d, %.2f VPk range,"
		      " %s coupled,\n", chan_list[i], range, couple_str))
	            DIAG_PRINTF(1, ("  measured %.3f VPk, %.3f VRMS, %.3f V"
		      " offset.\n", peak, sqrt(avg_sqr), offset))
	            DIAG_PRINTF(1, ("  Expected %.3f to %.3f VPk, %.3f to %.3f"
		      " VRMS, +- %.3f V offset\n", peak_min, peak_max,
	              sqrt(avg_sqr_min), sqrt(avg_sqr_max),
	              offset_max))
		}
	        sca = chan2sca[(chan_list[i] - 1) & 0xf];
		if ( ! fail_sca[sca] )  /* only one failure per sca */
		{
		    fail_sca[sca] = 1;
                    DIAG_Q_ERR(sca_err[sca]);
		}
		if ( errs == MAX_ERRS )
		{
	            DIAG_PRINTF(1, ("  ...and more errors\n"))
		}
	    }
	}

	/* increment, skip any that are not for this sca_type */
	while ( not_do_tbl[++loop] == sca_type );
    }
    if ( errs > 0 )
    {
	DIAG_PRINTF(0, ("  Input std in test failed.\n"))
	/* return the first sca to fail */
        for ( sca = 0; sca < E1432_SCAS; sca++ )
	{
	    if ( fail_sca[sca] ) return sca_err[sca];
	}
    }


cleanup: 
    if ( err )
    {
	DIAG_PRINTF(1, ("  Encoutered error %d.\n", err))
	DIAG_PRINTF(0, ("  Input std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }
    return err;
}


#define E1432_TACH_STATUS		(0x2200L * 4)
#define E1432_TACH_FIFO0		(0x2208L * 4)
#define E1432_TACH_FIFO1		(0x220cL * 4)
#define E1432_TACH_COUNT0		(0x2210L * 4)
#define E1432_TACH_COUNT1		(0x2214L * 4)

#define E1432_TACH_CHANS		2


static void
get_raw_tach_freq(E1432_MODULE_LIST_NODE *mn, SHORTSIZ16 chan, SHORTSIZ16 *err,
  FLOATSIZ32 tach_clock_freq, FLOATSIZ32 *freq)
{
    ULONGSIZ32 status = 0, count = 0, edge_0 = 0, edge_1 = 0;
    ULONGSIZ32 count_addr, fifo_addr;
    LONGSIZ32 delta;
    char *action;

    if ( chan )
    {
	count_addr = E1432_TACH_COUNT1;
	fifo_addr = E1432_TACH_FIFO1;
    }
    else
    {
	count_addr = E1432_TACH_COUNT0;
	fifo_addr = E1432_TACH_FIFO0;
    }

    action = "reading Tach status register";
    i1432_r32u(mn, err, E1432_TACH_STATUS, &status);
    if ( *err ) goto io_failure;

    action = "reading Tach count register";
    i1432_r32u(mn, err, count_addr, &count);
    if ( *err ) goto io_failure;

    count >>= 16;
    if ( (status & (1 << (16 + 2*chan))) || count < 2 )
    {
	*freq = 0.0f;
	return;
    }

    action = "reading Tach fifo register";
    i1432_r32u(mn, err, fifo_addr, &edge_0);
    i1432_r32u(mn, err, fifo_addr, &edge_1);
    if ( *err ) goto io_failure;
    delta = edge_1 - edge_0;
    if ( delta != 0 )
    {
	*freq =  tach_clock_freq / (FLOATSIZ32)delta;
    }
    else *freq = 0.0f;

    return;

io_failure:
    i1432_print_acc_fail(1, *err, action);
    *freq = 0.0f;
}

#define TPOS	E1432_TRIGGER_SLOPE_POS /* shorthand */
#define TNEG	E1432_TRIGGER_SLOPE_NEG /* shorthand */
#define F_LOW	1			/* measured frequency low */
#define F_HIGH	2			/* measured frequency low */
#define F_TOL	.2f			/* frequency tolerance */

static SHORTSIZ16
test_tach_std(E1432ID hw, struct e1432_hwconfig hwconfig)
{
    SHORTSIZ16 err = 0, errs = 0;
    SHORTSIZ16 cur_err, open_errs;
    SHORTSIZ16 attn_test = 0, dec_test = 0;
    SHORTSIZ16 nchan = hwconfig.tach_chans;
    SHORTSIZ16 chan_list[] = { E1432_TACH_CHAN(1), E1432_TACH_CHAN(2) };
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    SHORTSIZ16 group;
    SHORTSIZ16 i, chan;
    FLOATSIZ32 clock_freq = 51200.0f;
    FLOATSIZ32 tach_clock_freq;
    FLOATSIZ32 freq;
    FLOATSIZ32 trigger_level[] =
      {   0.0f,   0.8f,  -0.8f,   1.2f,  -1.2f,   0.0f,   8.0f,  0.0f };
    SHORTSIZ16 trigger_slope[] =
      {   TPOS,   TPOS,   TPOS,   TPOS,   TPOS,   TNEG,   TPOS,  TPOS };
    LONGSIZ32 tach_decimate[] =
      {      0,      0,      0,      0,      0,      0,      0,     4 };
    FLOATSIZ32 freq_targ[] =
      { 1.0e3f, 1.0e3f, 1.0e3f,   0.0f,   0.0f, 1.0e3f,   0.0f, .2e3f };
    SHORTSIZ16 tests = sizeof(trigger_level)/sizeof(FLOATSIZ32);
    FLOATSIZ32
      freq_meas[E1432_TACH_CHANS][sizeof(freq_targ)/sizeof(FLOATSIZ32)];
    SHORTSIZ16
      failures[E1432_TACH_CHANS][sizeof(freq_targ)/sizeof(FLOATSIZ32)];
    SHORTSIZ16 freq_zero[] = { 0, 0 };
    SHORTSIZ16 attn_errs[] = { 0, 0 };
    SHORTSIZ16 dec_errs[] = { 0, 0 };

    if ( nchan != E1432_TACH_CHANS )
    {
	DIAG_PRINTF(1, ("  Expected 2 tach channels, hwconfig indicates %d.\n",
	  nchan))
	DIAG_PRINTF(0, ("  Tach std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }

    DIAG_PRINTF(1, ("Testing Tach assuming 1 VPk, 1 KHz input.\n"))

    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
	DIAG_PRINTF(1, ("  e1432_create_channel_group returned %d\n", group))
	DIAG_PRINTF(0, ("  Tach std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }

    CHK(i1432_get_module_from_chan(hw, group, &mn));
    CHK(e1432_set_clock_freq(hw, group, clock_freq));
    CHK(e1432_get_tach_clock_freq(hw, group, &tach_clock_freq));
    CHK(e1432_set_trigger_mode(hw, group, E1432_TRIGGER_MODE_LEVEL));

    /* Turn off the tach polling in the 96k background - read tach values */
    i1432_w32(mn, &err, (LONGSIZ32)E1432_DIAG_MODES,
      (ULONGSIZ32)E1432_DIAG_MODE_NO_TACH_POLL);

    for ( i = 0; i < tests; i++ )
    {
        CHK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_LOWER,
	  trigger_level[i]));
        CHK(e1432_set_trigger_level(hw, group, E1432_TRIGGER_LEVEL_UPPER,
	  trigger_level[i]));
        CHK(e1432_set_trigger_slope(hw, group, trigger_slope[i]));
        CHK(e1432_set_tach_decimate(hw, group, tach_decimate[i]));
        /* Start measurement */
        CHK(e1432_init_measure(hw, group));
        i1432_pause(0.1);  /* for the measurement to start, get some edges */

	for ( chan = 0; chan < nchan; chan++ )
	{
	    cur_err = 0;
	    /* not able to get e1432_get_current_rpm() to work here */
            get_raw_tach_freq(mn, chan, &err, tach_clock_freq, &freq);
	    freq_meas[chan][i] = freq;
	    if ( err ) break;  /* bail out if something more serious */
	    if ( freq == 0 ) freq_zero[chan]++;
	    if ( freq_targ[i] == 0 && freq != 0 )
	    {
		cur_err = 1;
	    }
	    else if ( freq_targ[i] != 0 && freq == 0 )
	    {
		cur_err = 1;
	    }
	    else if ( freq < (1.0f - F_TOL) * freq_targ[i] )
	    {
		cur_err = 1;
	    }
	    else if ( freq > (1.0f + F_TOL) * freq_targ[i] )
	    {
		cur_err = 1;
	    }
	    if ( cur_err )
	    {
		errs++;
		failures[chan][i] = 1;
		if ( trigger_level[i] > 5.0f || trigger_level[i] < -5.0f )
		{
		    attn_errs[chan]++;
		}
		if ( tach_decimate[i] != 0 ) dec_errs[chan]++;
	    }
	    else failures[chan][i] = 0;
	}
    }

    if ( errs > 0 && ! err )
    /* handle measurement errors only if no gross failures */
    {
        if ( i1432_diag_print_level > 1 )
        {
	    if ( i1432_diag_print_level > 2 )
	    {
                (void) printf("    attn_errs = %d, %d\n", attn_errs[0], attn_errs[1]);
                (void) printf("    dec_errs  = %d, %d\n", dec_errs[0], dec_errs[1]);
                (void) printf("    freq_zero = %d, %d\n", freq_zero[0], freq_zero[1]);
                (void) printf("\n");
	    }
            (void) printf("    channnel  expected    received    trigger   "
	      " decimation\n");
            (void) printf("             frequency   frequency  level  slope"
	      "      ratio\n");
            for ( i = 0; i < tests; i++ )
            {
                for ( chan = 0; chan < nchan; chan++ )
                {
                    (void) printf("      %s %d %9.3f %11.3f %6.1f    %s %10ld\n",
                     failures[chan][i] ? "FAIL" : "    ",
		     chan, freq_targ[i], freq_meas[chan][i], trigger_level[i],
                     (trigger_slope[i] == TPOS) ? "pos" : "NEG" ,
		     tach_decimate[i] + 1);
                }
            }
        }
	open_errs = errs;
	if ( attn_errs[0] + attn_errs[1] == errs )
	/* only errors were with attenuators kicked in */
	{
            DIAG_PRINTF(1, ("  Failed to measure correct frequency\n"))
	    if ( attn_errs[0] == 0 || attn_errs[1] == 0 )
	    {
		if ( attn_errs[0] == 0 ) chan = 1;
		else chan = 0;
                DIAG_PRINTF(1, ("  only with chan %d attenuator on.\n",
		  chan + 1))
		DIAG_PRINTF(1, ("  Expected %.3f Hz, measured %.3f\n",
		  freq_targ[attn_test], freq_meas[chan][attn_test]))
                DIAG_PRINTF(1, ("  Suspect attenuator, drive.\n"))
	    }
	    else
	    {
                DIAG_PRINTF(1, ("  only with attenuators on, both channels.\n"))
		DIAG_PRINTF(1, ("  Expected %.3f Hz, measured %.3f (CH 1),"
		  " %.3f (CH 2)\n", freq_targ[attn_test],
		  freq_meas[0][attn_test], freq_meas[1][attn_test]))
                DIAG_PRINTF(1, ("  Suspect attenuators, drives.\n"))
	    }
	    open_errs = 0;  /* all diagnosed */
	}
	else if ( dec_errs[0] + dec_errs[1] == errs )
	/* only errors were with decimation/prescaling kicked in */
	{
            DIAG_PRINTF(1, ("  Failed to measure correct frequency\n"))
	    if ( dec_errs[0] == 0 || dec_errs[1] == 0 )
	    {
		if ( dec_errs[0] == 0 ) chan = 1;
		else chan = 0;
                DIAG_PRINTF(1, ("  only with chan %d decimation on.\n",
		  chan + 1))
		DIAG_PRINTF(1, ("  Expected %.3f Hz, measured %.3f\n",
		  freq_targ[dec_test], freq_meas[chan][attn_test]))
            }
	    else
	    {
                DIAG_PRINTF(1, ("  only with decimation on, both channels.\n"))
		DIAG_PRINTF(1, ("  Expected %.3f Hz, measured %.3f (CH 1),"
		  " %.3f (CH 2)\n", freq_targ[attn_test],
		  freq_meas[0][attn_test], freq_meas[1][attn_test]))
	    }
            DIAG_PRINTF(1, ("  Suspect LCA U12 and PROM U16.\n"))
	    open_errs = 0;  /* all diagnosed */
	}
	else if ( freq_zero[0] == tests || freq_zero[1] == tests )
	/* all measuremens for at least one channel received no edges */
	{
	    if ( freq_zero[0] == tests && freq_zero[1] == tests )
	    /* both channels dead */
	    {
                DIAG_PRINTF(1, ("  Never received any tach edges on either"
		  " channel.\n"))
                DIAG_PRINTF(1, ("  Suspect PLD U28, LCAs U12, U13, U14 and"
		  " PROMs U16, U15, U17.\n"))
	        open_errs = 0;  /* all diagnosed */
	    }
	    else
	    /* just one channel */
	    {
		if ( freq_zero[0] == tests ) chan = 0;
		else chan = 1;
                DIAG_PRINTF(1, ("  Never received any tach edges on channel"
		  " %d.\n", chan + 1))
                DIAG_PRINTF(1, ("  Suspect analog signal path, comparators.\n"))
                /* these errors now diagnosed, mark them as being OK */
                for ( i = 0; i < tests; i++ )
		{
		    if ( freq_targ[i] != 0.0f )
		    {
			failures[chan][i] = 0;
			open_errs--;
		    }
		}
	    }
	}
        if ( freq_zero[0] == 0 && freq_zero[1] == 0 )
        /* both channels broken */
        {
            DIAG_PRINTF(1, ("  Always received tach edges on each channel,\n"))
            DIAG_PRINTF(1, ("  regardless of trigger level settings.\n"))
            DIAG_PRINTF(1, ("  Front end DACs, attenuators probably not"
	      " getting programmed.\n"))
            DIAG_PRINTF(1, ("  Suspect DAC U3 and programming lines\n"))
            DIAG_PRINTF(1, ("  SERREGn, SERCLK, SERDAT, SERREGn.\n"))
            /* these errors now diagnosed, mark them as being OK */
            for ( i = 0; i < tests; i++ )
	    {
	        if ( freq_targ[i] == 0.0f )
	        {
		    failures[chan][i] = 0;
		    open_errs--;
	        }
	    }
        }
	if ( open_errs > 0 )
	{
            DIAG_PRINTF(1, ("  Some tach tests failed.\n"))
	}
    }

    if ( err || errs > 0 )
    {
        DIAG_PRINTF(0, ("  Tach std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_TACH);
    }

cleanup: 
    if ( err )
    {
	DIAG_PRINTF(1, ("  Encoutered error %d.\n", err))
	DIAG_PRINTF(0, ("  Tach std in test failed.\n"))
        return DIAG_Q_ERR(ERR1432_DIAG_ERR_UNDIAGNOSED);
    }
    return err;
}


/***************************************************************************
 All selftests are rolled into this function and accessed through the
 various bits of the parameter, flags.
 ***************************************************************************/   
SHORTSIZ16 EXPORT
i1432_selftest(SHORTSIZ16 la, SHORTSIZ16 tests, SHORTSIZ16 main,
  SHORTSIZ16 *mem, SHORTSIZ16 *scas)
{
    SHORTSIZ16 tmp_err, err = 0;
    SHORTSIZ16 sca, error2;
    E1432_MODULE_LIST_NODE node;
    E1432_MODULE_LIST_NODE *mn = &node;
    struct i1432_chan_info tmp_chan_list;
    struct i1432_chan_info *save_chan_list;
    int save_chan_count = -1;
    E1432ID hw = 0;
    SHORTSIZ16 save_print_level = i1432_diag_print_level;
    const char *save_install_file = i1432_install_file;

    DIAG_PRINTF(2,
      ("  i1432_selftest(%d, %d, %d, %d, {%d, %d, %d, %d, %d})\n",
      la, tests, main, *mem,
      scas[0], scas[1], scas[2], scas[3], scas[4], scas[5]))

    /* Initialize library things */
    err = e1432_init_io_driver();
    /* restore the print_level, install_file, since e1432_init_io_driver()
       clears them */
    i1432_diag_print_level = save_print_level;
    i1432_install_file = save_install_file;
    if ( err )
    {
	DIAG_PRINTF(0, ("  VXI test failed - interface failure,"
	  " probably not a module problem.\n"));
	return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);   
    }

    e1432_set_try_recover(1);

    if ( tests & E1432_SELFTEST_NON_DL )
    {
         /* Do this before calling i1432_fake_setup_sicl */
#if ( defined(HAVE_SICL) || defined(HAVE_VTL) )
        mn->sicl_id = 0;
#endif
        mn->a24_base = NULL;
        save_chan_list = i1432_chan_list[0];
        save_chan_count = i1432_chan_count[0];
        err = i1432_fake_setup_sicl(mn, la, &tmp_chan_list,
	  E1432_EXTRA_IO_DIAG);
	if ( err )
	{
	    /* I don't know why this doesn't just do a goto cleanup,
	       but I'm afraid to change it now. */
	    DIAG_PRINTF(0, ("  Main board or VXI failure.\n"));
	    (void) i1432_fake_cleanup_sicl(mn, save_chan_list,
					   save_chan_count);
	    e1432_set_try_recover(0);
	    (void) e1432_uninit_io_driver();
	    return DIAG_Q_ERR(ERR1432_DIAG_ERR_SUBST);
	}
	if ( mn->a24_256k == 0 )
	{
	    /* warning that U707 should be updated if given a chance */
	    DIAG_PRINTF(1, ("  Note: non 256k VXI interface found.\n"));
        }

	if ( main & E1432_SELFTEST_VXI ) CHK(i1432_test_vxi(mn))
	/* need to reset module one way or the other */
        else i1432_berr_recover(mn);
        if ( main & E1432_SELFTEST_BSRAM ) CHK(i1432_test_bsram(mn));
        if ( main & E1432_SELFTEST_DSP ) CHK(i1432_test_vxi_dsp(mn));
        if ( main & E1432_SELFTEST_96K_HOSTPORT )
	{
	     CHK(i1432_test_96k_hostport(mn));
	}
        if ( main & E1432_SELFTEST_ASRAM ) CHK(i1432_test_asram(mn));
        CHK(i1432_test_dram(mn, mem));
        if ( main & E1432_SELFTEST_PLL ) CHK(i1432_test_pll(mn));
        if ( main & E1432_SELFTEST_FLASH ) CHK(i1432_test_flash(mn));

	/* check out what SCAs, FIT devices present */
	/* updated the entries in scas[] as able */
        for ( sca = 0; sca <  E1432_SCAS; sca++ )
        {
	    /* force testing of second connector of E1432 scas */
	    if ( sca > 1 && sca < E1432_SCA_FIT
	      && (scas[sca-2] == E1432_SCA_ID_SONATA_A
	      || scas[sca-2] == E1432_SCA_ID_SONATA_B) )
	    {
		scas[sca] = scas[sca-2];
	    }
            tmp_err = i1432_check_sca(mn, sca, &scas[sca]);
	    if ( tmp_err ) err = tmp_err;
        }
        DIAG_PRINTF(2, ("  found: %d {%d, %d, %d, %d, %d})\n",
          *mem, scas[0], scas[1], scas[2], scas[3], scas[4]))
        if ( err ) goto cleanup;

	/* SCA/FIT testing */
	for ( sca = 0; sca < E1432_SCAS; sca++ )
	{
            switch( scas[sca] )
            {
            case E1432_SCA_ID_VIBRATO:
		tmp_err = i1432_test_vibrato(mn, sca);
		break;
            case E1432_SCA_ID_SONATA_A:
		tmp_err = i1432_test_sonata_a(mn, sca);
		break;
            case E1432_SCA_ID_SONATA_B:
		tmp_err = i1432_test_sonata_b(mn, sca);
		break;
            case E1432_SCA_ID_TACH:
		tmp_err = i1432_test_tach(mn);
		break;
            case E1432_SCA_ID_CLARION:
            case E1432_SCA_ID_CLARINET:
		tmp_err = i1432_test_src(mn, sca, scas[sca]);
		break;
	    }
	    if ( tmp_err ) err = tmp_err;
	}
        if ( err ) goto cleanup;

	/* reset the page map register */
        (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
    }

    /* need to wait on cleaning up sicl until after download */
    if ( (tests & E1432_SELFTEST_DL) || (tests & E1432_SELFTEST_STD_IO) )
    {
        DIAG_PRINTF(1, ("Performing download testing.\n"))
	/* if there are any bootup errors, they are evaluated below,
	   by i1432_map_errors() */
        CHK(i1432_preinstall(la));
    }
    if ( tests & E1432_SELFTEST_NON_DL )
    {
        CHK(i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count));
    }

    if ( (tests & E1432_SELFTEST_DL) || (tests & E1432_SELFTEST_STD_IO) )
    {
	int     src_chans;
        struct e1432_hwconfig hwconfig;

        CHK(e1432_assign_channel_numbers(1, &la, &hw));
        CHK(e1432_get_hwconfig(1, &la, &hwconfig));
        src_chans = hwconfig.source_chans;

	if ( tests & E1432_SELFTEST_DL )
	{
	    /* source testing */
	    for ( sca = 0; sca < E1432_SCAS; sca++ )
	    {
	        if ( scas[sca] == E1432_SCA_ID_CLARION
	          || scas[sca] == E1432_SCA_ID_CLARINET )
	        {
	            CHK(i1432_test_src_dld(hw, sca, src_chans));
	        }
	    }
        }

	if ( tests & E1432_SELFTEST_STD_IO )
	{
	    /* input channels */
            if ( hwconfig.input_chans > 0 )
	    {
                CHK(i1432_test_in_std(hw, hwconfig, scas));
	    }
	    if ( scas[E1432_SCA_FIT] == E1432_SCA_ID_TACH )
	    {
                CHK(test_tach_std(hw, hwconfig));
	    }
	    /* source testing */
	    for ( sca = 0; sca < E1432_SCAS; sca++ )
	    {
	        if ( scas[sca] == E1432_SCA_ID_CLARION
	          || scas[sca] == E1432_SCA_ID_CLARINET )
	        {
	            CHK(i1432_test_src_std(hw, sca, src_chans));
	        }
	    }
	}
    }

    DIAG_PRINTF(1, ("done.\n"))

cleanup: 
    if ( err )
    {
        err = i1432_map_errors(err, la, mn);
	DIAG_PRINTF(2, ("  e1432_selftest() returning %d, %s\n",
          err, i1432_get_error_string(err)))
        DIAG_PRINTF(1, ("  Selftest failed.\n"))
    }
    if ( i1432_chan_list[0] != save_chan_list )
    {
	/* reset the page map register */
        (void) i1432_direct_write_register(mn, E1432_PAGE_MAP_REG, 0);
        error2 = i1432_fake_cleanup_sicl(mn, save_chan_list, save_chan_count);
	if (error2)
	    err = error2;
    }
    e1432_set_try_recover(0);
    (void) e1432_uninit_io_driver();
    return err;
}


SHORTSIZ16 EXPORT
e1432_selftest(SHORTSIZ16 mods, SHORTSIZ16 *las, SHORTSIZ16 tests, void *opts)
{
    int i;

    if ( mods == 0 ) return ERR1432_ILLEGAL_MODULE_COUNT;
    else if ( mods > 1 )
    {
	SHORTSIZ16 tmp_err, err = 0;
	for ( i = 0; i < mods; i++ )
	{
	    tmp_err = e1432_selftest(1, &las[i], tests,
	      (void *)((char **)opts)[i]);
	    if ( tmp_err ) err = tmp_err;
	}
        return err;
    }
    else
    {
	SHORTSIZ16 main = E1432_SELFTEST_MAIN;
	SHORTSIZ16 mem, scas[E1432_SCAS];
	mem = i1432_dram_size((char *)opts);
	for ( i = 0; i < E1432_SCAS; i++ )
	{
	    scas[i] = i1432_sca_type(i, (char *)opts);
	}
        return i1432_selftest(*las, tests, main, &mem, scas);
    }
}


SHORTSIZ16 EXPORT
e1432_set_diag_print_level(SHORTSIZ16 print_level)
{
    i1432_diag_print_level = print_level;
    return 0;
}
